% epsilon_estimate_iv.m
% 
% estimate epsilons via IV regression
% 
% "The Past and Future of U.S. Structural Change" 
% Andrew Foerster, Andreas Hornstein, Pierre-Daniel Sarte, Mark Watson
% September 2025
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

% -- Clear Workspace -- %
restoredefaultpath;
clear;
clc;
close all;

% -- File Directories -- % 
datadir = 'Data\DataReplication\DataFinal\'; 
outdir  = 'Tables\';
figdir  = 'Figures\';
matdir  = 'Output\';

% -- Add paths -- %
addpath('Functions/');
addpath('Utilities/')
addpath('Data/')

% Load Data
load_data;

period_lower = 20;       % Shortest Period 
Share_mat = sV_mat;
T = size(Share_mat,1);
% Compute q
q = floor(2*T/period_lower);

% Log Prices
py = y ./ yq;
log_py = log(py);
d_p = dif(log_py,1);

% Log Shares of Investment
log_omega = log(omega);
% Log Shares of Materials
log_phi = log(phi);

% Durables and IPP in Sector 
P_Dur = py(1:T,2);
P_IPP = py(1:T,3);
S_X_Dur = squeeze(omega(2,:,:))';
S_X_IPP = squeeze(omega(3,:,:))';
S_M_Dur = squeeze(phi(2,:,:))';
S_M_IPP = squeeze(phi(3,:,:))';
X_Dur_IPP = (S_X_Dur./repmat(P_Dur,1,5))./(S_X_IPP./repmat(P_IPP,1,5)); % Relative quantities of dur/ipp in investment
M_Dur_IPP = (S_M_Dur./repmat(P_Dur,1,5))./(S_M_IPP./repmat(P_IPP,1,5)); % Relative quantities of dur/ipp in materials   
P_Dur_IPP = P_Dur./P_IPP; % Relative prices of dur/ipp

% Compute logs 
x_Dur_IPP = log(X_Dur_IPP);
m_Dur_IPP = log(M_Dur_IPP);
p_Dur_IPP = log(P_Dur_IPP);
% Compute first differences
d_x_Dur_IPP = dif(x_Dur_IPP,1);
d_m_Dur_IPP = dif(m_Dur_IPP,1);
d_p_Dur_IPP = dif(p_Dur_IPP,1);
d_a_Dur_IPP = d_tfp(:,2)-d_tfp(:,3); % Relative TFP growth rates

% Drop first observation, which is missing
d_x_Dur_IPP = d_x_Dur_IPP(2:end,:);
d_m_Dur_IPP = d_m_Dur_IPP(2:end,:);
d_p_Dur_IPP = d_p_Dur_IPP(2:end,:);
d_a_Dur_IPP = d_a_Dur_IPP(2:end,:);
calvec = calvec(2:end);
calds = calds(2:end,:);
T = length(calvec);

% Get Cosine transforms
[cx,~,~,~] = cos_transform(d_x_Dur_IPP,q);
[cm,~,~,~] = cos_transform(d_m_Dur_IPP,q);
[cp,~,~,~] = cos_transform(d_p_Dur_IPP,q);
[ca,~,~,~] = cos_transform(d_a_Dur_IPP,q);

% Statistics to compute
% OLS Coefficients:
    % 1. -q (x or m) on a
    % 2. p on a
    % 3. 4 element covariance matrix of 1 and 2
% IV Coefficients:
    % 1. -q on p using a as instrument
    % 2. se of (1)
% AR confidence set for eps
    % 1. lower bound
    % 2. upper bound

% Set up Matrices to store results
rslt_ols_m = NaN(6,5);
rslt_iv_m = NaN(2,5);
rslt_ar_m = NaN(2,5);
rslt_ols_x = NaN(6,5);
rslt_iv_x = NaN(2,5);
rslt_ar_x = NaN(2,5);

% Set up Grid of value for epsilon
% eps_min = 0.001;
eps_min = -20.0;
eps_max = 20;
n_grid = 5000;
eps_grid = linspace(eps_min,eps_max,n_grid)';

% Loop over sectors
for i = 1:5
    x = cx(:,i);
    m = cm(:,i);
    p = cp;
    a = ca;
    % OLS Results
    [b_ols_x,s_ols_x] = ols_rslt(x,p,a);
    [b_ols_m,s_ols_m] = ols_rslt(m,p,a);
    rslt_ols_x(:,i) = [b_ols_x;s_ols_x(:)];
    rslt_ols_m(:,i) = [b_ols_m;s_ols_m(:)];
    % IV Results
    [b_iv_x,se_b_iv_x] = iv_rslt(x,p,a);
    [b_iv_m,se_b_iv_m] = iv_rslt(m,p,a);
    rslt_iv_x(:,i) = [b_iv_x;se_b_iv_x];
    rslt_iv_m(:,i) = [b_iv_m;se_b_iv_m];
    % AR confidence set
    pval_m = NaN(n_grid,1);
    pval_x = NaN(n_grid,1);
    for i_grid = 1:n_grid
        eps = eps_grid(i_grid);
        pval_m(i_grid) = ar_test(m,p,a,eps);
        pval_x(i_grid) = ar_test(x,p,a,eps);
    end
    ii = pval_m > 0.33;
    e = eps_grid(ii==1);
    rslt_ar_m(1,i) = min(e);
    rslt_ar_m(2,i) = max(e);
    ii = pval_x > 0.33;
    e = eps_grid(ii==1);
    rslt_ar_x(1,i) = min(e);
    rslt_ar_x(2,i) = max(e);
end

% Save Results
ofile = [outdir 'eps_estimates_iv_' num2str(q) '.csv'];
fid = fopen(ofile, 'w');

% Save Results
fprintf(fid,'Results for estimating elasticity of substitution using IV regressions \n');
fprintf(fid,'q = %d\n',q);
% fprintf(fid,'Date: %s\n',this_date);
fprintf(fid,'\n\n');
fprintf(fid,'Results for Material Inputs\n');
fprintf(fid,'Sector,,AR Confidence Set,,,IV Estimates,,,OLS Estimates\n');
fprintf(fid,',,lower,upper,,eps_hat,2SLS_se(eps_hat),,bhat(-q on a),bhat(p on a), sebhat(-q on a), sebhat(p on a), corr(bhats)\n');
for i = 1:5
    tmp = char(SecName(i));
    fprintf(fid,[tmp ',,']);
    fprintf(fid,'%5.2f,%5.2f,,',rslt_ar_m(1,i),rslt_ar_m(2,i));
    fprintf(fid,'%5.2f,%5.2f,,',rslt_iv_m(1,i),rslt_iv_m(2,i));
    se_1 = sqrt(rslt_ols_m(3,i));
    se_2 = sqrt(rslt_ols_m(6,i));
    cor = rslt_ols_m(5,i)/(se_1*se_2);
    fprintf(fid,'%5.2f,%5.2f,%5.2f,%5.2f,%5.2f',rslt_ols_m(1,i),rslt_ols_m(2,i),se_1,se_2,cor);
    fprintf(fid,'\n');
end

fprintf(fid,'\n\n');
fprintf(fid,'Results for Investment Inputs\n');
for i = 1:5
    tmp = char(SecName(i));
    fprintf(fid,[tmp ',,']);
    fprintf(fid,'%5.2f,%5.2f,,',rslt_ar_x(1,i),rslt_ar_x(2,i));
    fprintf(fid,'%5.2f,%5.2f,,',rslt_iv_x(1,i),rslt_iv_x(2,i));
    se_1 = sqrt(rslt_ols_x(3,i));
    se_2 = sqrt(rslt_ols_x(6,i));
    cor = rslt_ols_x(5,i)/(se_1*se_2);
    fprintf(fid,'%5.2f,%5.2f,%5.2f,%5.2f,%5.2f',rslt_ols_x(1,i),rslt_ols_x(2,i),se_1,se_2,cor);
    fprintf(fid,'\n');
end


% Functions 
function [b,s] = ols_rslt(q,p,a)
    % OLS Coefficients
    y = [-q p];
    b = a\y;
    u = y - a*b;
    s = (u'*u/(length(a)-1));
    b = b';
end

function [b,se_b] = iv_rslt(q,p,a)
    b = -(a'*q)/(a'*p);
    u = q - p*b;
    s = (u'*u/(length(a)-1));
    vb = s*(a'*a)/(a'*p)^2;
    se_b = sqrt(vb);
end

function pval = ar_test(q,p,a,eps)
    y = q+eps*p;
    x = a;
    b = x\y;
    u = y - x*b;
    s = (u'*u/(length(x)-1));
    vb = s/(x'*x);
    F = b^2/vb;
    pval = 1-fcdf(F,1,length(x)-1);
end
